import 'package:user/model/user_response.dart';
import 'package:dio/dio.dart';
import 'package:user/utils/logging_interceptor.dart';

class UserApiProvider {
  final String _endpoint = "https://randomuser.me/api/";
  Dio _dio;

  UserApiProvider() {
    print('UserApiProvider constructor');
    BaseOptions options =
        BaseOptions(receiveTimeout: 5000, connectTimeout: 5000);
    print('UserApiProvider - create dio instance');
    _dio = Dio(options);
    print('UserApiProvider - dio interceptors add a LoggingInterceptor');
    _dio.interceptors.add(LoggingInterceptor());
  }

  Future<UserResponse> getUser() async {
    print('UserApiProvider - getUser');
    try {
      return Future.delayed(const Duration(milliseconds: 5000), () async {
        Response response = await _dio.get(_endpoint);
        print('√√√√ response received!');
        return UserResponse.fromJson(response.data);
      });
    } catch (error, stacktrace) {
      print(
          "UserApiProvider - Exception occured: $error stackTrace: $stacktrace");
      return UserResponse.withError(_handleError(error));
    }
  }

  String _handleError(Error error) {
    String errorDescription = "";
    if (error is DioError) {
      DioError dioError = error as DioError;
      switch (dioError.type) {
        case DioErrorType.CANCEL:
          errorDescription = "Request to API server was cancelled";
          break;
        case DioErrorType.CONNECT_TIMEOUT:
          errorDescription = "Connection timeout with API server";
          break;
        case DioErrorType.DEFAULT:
          errorDescription =
              "Connection to API server failed due to internet connection";
          break;
        case DioErrorType.RECEIVE_TIMEOUT:
          errorDescription = "Receive timeout in connection with API server";
          break;
        case DioErrorType.RESPONSE:
          errorDescription =
              "Received invalid status code: ${dioError.response.statusCode}";
          break;
        case DioErrorType.SEND_TIMEOUT:
          errorDescription = "Send timeout in connection with API server";
          break;
      }
    } else {
      errorDescription = "Unexpected error occured";
    }
    return errorDescription;
  }
}
