


import UIKit

class ViewController: UIViewController {
    
    var button : UIButton? = nil
    
    let dispatchQueue = DispatchQueue(label: "com.prit.TestGCD.DispatchQueue",
                                      attributes: .concurrent)
    

    //https://developer.apple.com/documentation/dispatch/dispatchsemaphore/1452955-init
    
    // value is thestarting value for the semaphore. If its 1, means it can ONLY let 1 in at the same time.
    // 2 means, 2 can use this semphore at the same time.
    
    // Reader and Writers fight the resource semaphore
    let semaphoreResource = DispatchSemaphore(value: 1)
    
    // Readers fight for the reader mutex in order to update readCount. 
    // readCount then decides whether the reader should lock the resource semaphore or not.
    let semaphoreReaderMutex = DispatchSemaphore(value:1)
    var readCount = 0
    
    var runReadCodeBlock: ((String)->())?
    var runWriteCodeBlock: ((String)->())?
    
    override func viewDidLoad() {

        runReadCodeBlock = { processName in
            print("\(processName) reading stuff..")
            Thread.sleep(forTimeInterval: 3)
            print("\(processName) done...")
        }
        
        runWriteCodeBlock = { processName in
            print("\(processName) writing stuff....please wait")
            Thread.sleep(forTimeInterval: 5)
            print("\(processName) all done!")
        }
    }
    
    override func viewDidAppear(_ animated: Bool) {
        super.viewDidAppear(animated)
        
        let dispatchGroup = DispatchGroup()
        
        (DispatchQueue.global(qos: .userInitiated)).async(group: dispatchGroup, qos: .userInitiated, flags: .assignCurrentContext) {
            () -> Void
            in
            self.reader(name: "readerA")
            self.reader(name: "readerB")
        }
        
        (DispatchQueue.global(qos: .userInitiated)).async(group: dispatchGroup, qos: .userInitiated, flags: .assignCurrentContext) {
            () -> Void
            in
            self.reader(name: "readerC")
        }
        
        (DispatchQueue.global(qos: .userInitiated)).async(group: dispatchGroup, qos: .userInitiated, flags: .assignCurrentContext) {
            () -> Void
            in
            self.writer("writerA")
        }
        
        (DispatchQueue.global(qos: .userInitiated)).async(group: dispatchGroup, qos: .userInitiated, flags: .assignCurrentContext) {
            () -> Void
            in
            self.reader(name: "readerD")
        }
        
        (DispatchQueue.global(qos: .userInitiated)).async(group: dispatchGroup, qos: .userInitiated, flags: .assignCurrentContext) {
            () -> Void
            in
            self.reader(name: "readerE")
        }
        
        (DispatchQueue.global(qos: .userInitiated)).async(group: dispatchGroup, qos: .userInitiated, flags: .assignCurrentContext) {
            () -> Void
            in
            self.reader(name: "readerF")
            self.reader(name: "readerG")
        }
        
 
    }
    
    
    
    private func grabResource(semaphore : DispatchSemaphore, processName : String) -> Bool {
        if semaphore.wait(timeout: .distantFuture) == .success {
            print("\(processName) grabs the \(semaphore.description) semaphore")
            return true
        } else {
            print("\(processName) waiting for \(semaphore.description) timed out")
            return false
        }
    }
    
    private func letGoResource(semaphore : DispatchSemaphore, name : String) {
            print("\(name) lets go of the \(semaphore.description)")
            semaphore.signal()
    }
    
    public func writer(_ name : String) {
        if grabResource(semaphore: semaphoreResource, processName: name) {
            runWriteCodeBlock?(name)
            letGoResource(semaphore: semaphoreResource, name: name)
        }
    }
    
    
    // the idea is that we grab the reader semaphore in order to make changes to the resource semaphore
    // the reader semaphore allows the 1st reader to lock the resource semaphore, and naturally, the last reader to unlock the semaphore
    
    // Any readers after the 1st one, does not need to lock the resource semaphore anymore. They just go on ahead and read from the resource.
    // However, they DO NEED to grab the reader mutex because they are changing the readCount variable. This is to update the number of readers.
    // Only when the last reader finsihes reading, updates the readCount to 0, then it unlocks the resource semaphore so that
    // the writers can write.
    
    public func reader( name : String) {
        
        // grab reader mutex so that we can increment the readCount var
        if grabResource(semaphore: semaphoreReaderMutex, processName: name) {
            readCount += 1
            if readCount == 1 { // if we're the first reader, we must lock the resource mutex from Writers.
                // the reason why we don't lock from Readers is that this function ensures Readers can go through
                _ = grabResource(semaphore: semaphoreResource, processName: name)
            }
            
            // we're not the first reader, the resource mutex was already grabbed. So 2nd...n readers simply just read
        }
        letGoResource(semaphore: semaphoreReaderMutex, name: name)
        
        runReadCodeBlock?(name)

        // grab reader mutex, because we're gunna decrease the readCount
        if grabResource(semaphore: semaphoreReaderMutex, processName: name) {
            readCount -= 1
            if readCount == 0 {
                // we're the last reader, we must release the resource semaphore
                letGoResource(semaphore: semaphoreResource, name: name)
            }
        }
        letGoResource(semaphore: semaphoreReaderMutex, name: name)
    }
    
    
    override func didReceiveMemoryWarning() {
        super.didReceiveMemoryWarning()
        // Dispose of any resources that can be recreated.
    }
}





